<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\User;
use App\OfertaProveedor;
use App\ProductoProveedor;
use App\Municipio;
use App\StockProveedor;
use App\Participante;
use App\Carrito;
use App\CarritoProveedor;
use App\CarritoProducto;

class OfertaProveedorController extends Controller
{
    public function __construct(){
        $this->middleware('jwt.auth');
    }

    /**
     * Retorna todos los productos que tienen ofertas pendientes y del usuario autenticado.
     */
    public function index(){
        return CarritoProveedor::where('id_proveedor', auth()->user()->id)
                ->with(['carrito' => function ( $carrito ){
                    $carrito->with('usuario');
                }])
                ->get();
    }

    public function totalOfertas(){
        $productos = ProductoProveedor::withCount([
            'ofertas as ofertas_pendientes' => function ( $oferta ) {
                $oferta->where('estado', 'EN PROCESO');
            }
        ])->having('ofertas_pendientes', '>', 0)
            ->having('id_user', auth()->user()->id)->get();

        return count( $productos );
    }

    /**
     * Muestra todas las ofetas de un unico producto
     */

    public function show( Request $request ) {
        $data = $request->all();
        return CarritoProveedor::where('id_proveedor', auth()->user()->id)->where('id_carrito', $data['id_carrito'])
                ->with(['carrito' => function ( $carrito ){
                    $carrito->with(['usuario' => function( $usuario ){
                        $usuario->with('municipio');
                    }]);
                }, 'carritoProductos' => function( $carritoProducto ){
                    $carritoProducto->with(['productoProveedor' => function( $productoProveedor ){
                        $productoProveedor->with(['tipoProductoProveedor' => function( $tipoProducto ){
                            $tipoProducto->with('unidad');
                        }, 'stocks_proveedores']);
                    }, 'oferta']);
                }])
                ->first();
    }

    public function store(Request $request){
        $data = $request->all();

        // Actualiza estado del carrito
        $carrito = Carrito::find($data['id']);
        $carrito->estado = 'ENVIADO';
        $carrito->valor_total = $data['valor_total'];

        $carrito->save();

        $oferta['id_user'] = auth()->user()->id;
        $oferta['id_municipio_destino'] = auth()->user()->id_municipio;
        $oferta['direccion_destino'] = auth()->user()->direccion;

        foreach ($data['carrito_proveedores'] as $key => $carritoProveedor) {

            $oferta['id_municipio_origen'] = $carritoProveedor['proveedor']['id_municipio'];
            $oferta['latitude'] = $carritoProveedor['proveedor']['latitude'];
            $oferta['longitude'] = $carritoProveedor['proveedor']['longitude'];

            foreach ($carritoProveedor['carrito_productos'] as $key => $carritoProducto) {
                $oferta['id_carrito_producto'] = $carritoProducto['id'];
                $oferta['cantidad'] = $carritoProducto['cantidad'];
                $oferta['id_producto'] = $carritoProducto['producto_proveedor']['id'];
                $oferta['valor_unitario'] = $carritoProducto['producto_proveedor']['stocks_proveedores'][ count($carritoProducto['producto_proveedor']['stocks_proveedores']) - 1 ]['valor_stock'];
                $oferta['valor_total'] = $oferta['cantidad'] * $oferta['valor_unitario'];
                $oferta['id_stock'] = $carritoProducto['producto_proveedor']['stocks_proveedores'][ count($carritoProducto['producto_proveedor']['stocks_proveedores']) - 1 ]['id'];

                OfertaProveedor::create($oferta);
            }
        }

        return response()->json([
            'susses' => true,
            'message' => 'Oferta Realizada Exitosamente',
            'codigo' => 200
        ], 200);

        // return $data;

    }

    public function informacionOferta( $id ){
        $oferta = Oferta::where('id', $id)
            ->with([
                'producto' => function ( $producto ) {
                    $producto->with('tipoProducto');
                },
                'usuario',
                'municipioOrigen',
                'municipioDestino',

            ])
            ->first();
        
        if($oferta['usuario']['id'] === auth()->user()->id || auth()->user()->id_role === 2){
            $oferta['vendedor'] = User::where('id', $oferta['producto']['id_user'])->first();
            $oferta['stock_oferta'] = Stock::where('id', $oferta['id_stock'])->first();
            $oferta['es_participante'] = Participante::where('id_user', $oferta['id_user'])->first() == null ? 'NO' : 'SI';
            return $oferta;
        }

        return response()->json([
            'susses' => false,
            'message' => 'Solicitud no permitida.',
            'codigo' => 403
        ], 403);


    }

    public function update( Request $request ) {
        $data = $request->all();

        // Obtiene la oferta a actualizar
        $oferta = OfertaProveedor::find($data['id_oferta']);

        if($data['nuevo_estado'] === 'ACEPTADO'){

            // Verifica si se ha cambiado la cantidad de la oferta
            if($data['cantidad_alterada'] === 'No'){
                $oferta->cantidad_venta = $data['nueva_cantidad'];
            }else{
                $oferta->cantidad_venta = $oferta->cantidad;
            }

            // Verifica si se ha cambiado el valor unitario de la oferta
            if($data['valor_alterado'] === 'No'){
                $oferta->valor_unitario_venta = $data['nuevo_valor'];
                $oferta->valor_total_venta = $data['nuevo_valor'] * $oferta->cantidad_venta;
            }else{
                $oferta->valor_unitario_venta = $oferta->valor_unitario;
                $oferta->valor_total_venta = $oferta->valor_unitario * $oferta->cantidad_venta;
            }

            $oferta->estado = 'ACEPTADO';
            $oferta->metodo_pago = $data['metodo_pago'];
            $oferta->metodo_envio = $data['metodo_envio'];
            $oferta->valor_envio = $data['valor_envio'];

            // Actualizar la oferta
            $oferta->save();

            // Actualizar el valor del stock
            $stocks = StockProveedor::where('id_producto', $data['id_producto'])->get();

            $valor_ultimo_stock = $stocks[count($stocks)-1]['valor_stock'];
            StockProveedor::create( [ 'stock' => $data['nuevo_stock'], 'valor_stock' => $valor_ultimo_stock, 'id_producto' => $data['id_producto'] ] );

            $producto = ProductoProveedor::find($data['id_producto']);
            $producto->stock_actual = $data['nuevo_stock'];
            $producto->save();
            
        }else if($data['nuevo_estado'] === 'RECHAZADO'){

            $oferta->estado = 'RECHAZADO';
            $oferta->recibido = 'NO';
            // Actualizar la oferta
            $oferta->save();

        }

        // Obtiene la información del producto actualizada
        return CarritoProveedor::where('id_proveedor', auth()->user()->id)->where('id_carrito', $data['id_carrito'])
                ->with(['carrito' => function ( $carrito ){
                    $carrito->with(['usuario' => function( $usuario ){
                        $usuario->with('municipio');
                    }]);
                }, 'carritoProductos' => function( $carritoProducto ){
                    $carritoProducto->with(['productoProveedor' => function( $productoProveedor ){
                        $productoProveedor->with(['tipoProductoProveedor' => function( $tipoProducto ){
                            $tipoProducto->with('unidad');
                        }, 'stocks_proveedores']);
                    }, 'oferta']);
                }])
                ->first();

    }

    public function ofertasRealizadas(){
        return Oferta::with(['producto' => function ( $producto ){
                    $producto->with(['tipoProducto' => function ( $tipo ){
                        $tipo->with('imagenes');
                    }]);
                }])
                ->where('id_user', auth()->user()->id)->get();
    }

    public function respuestas(){
        return count(Oferta::where('id_user', auth()->user()->id)
                ->whereIn('estado', ['RECHAZADO', 'ACEPTADO'])
                ->where('visto', 'NO')->get());
    }

    public function visado(Request $request){
        $data = $request->all();

        $oferta = Oferta::find($data['id']);

        $oferta->visto="SI";

        $oferta->save();

        return count(Oferta::where('id_user', auth()->user()->id)
                ->whereIn('estado', ['RECHAZADO', 'ACEPTADO'])
                ->where('visto', 'NO')->get());
    }

    public function compraRecibida(Request $request){
        $data = $request->all();

        $oferta = Oferta::find($data['oferta']);

        if($oferta['estado'] !== 'RECHAZADO'){
            $oferta->latitude = $data['latitude'];
            $oferta->longitude = $data['longitude'];
            $oferta->recibido = 'SI';

            $oferta->save();

            $oferta->fecha_recibido = $oferta['updated_at'];

            $oferta->save();
        }

        return Oferta::with(['producto' => function ( $producto ){
            $producto->with(['tipoProducto' => function ( $tipo ){
                $tipo->with('imagenes');
            }]);
        }])
        ->where('id_user', auth()->user()->id)->get();
    }

}
