<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\User;
use App\OfertaInclusionProductiva;
use App\ProductoInclusionProductiva;
use App\Municipio;
use App\StockInclusionProductiva;
use App\Participante;

class OfertaInclusionProductivaController extends Controller
{
    public function __construct(){
        $this->middleware('jwt.auth');
    }

    /**
     * Retorna todos los productos que tienen ofertas pendientes y del usuario autenticado.
     */
    public function index(){

        return ProductoInclusionProductiva::with(['tipoProducto' => function ( $tipo ){
                $tipo->with('imagenes');
            }])->withCount([
                'ofertas as ofertas_pendientes' => function ( $oferta ) {
                    $oferta->where('estado', 'EN PROCESO');
                }
            ])->having('ofertas_pendientes', '>', 0)
                ->having('id_user', auth()->user()->id)->get();

    }

    public function totalOfertas(){

        $productos = ProductoInclusionProductiva::withCount([
            'ofertas as ofertas_pendientes' => function ( $oferta ) {
                $oferta->where('estado', 'EN PROCESO');
            }
        ])->having('ofertas_pendientes', '>', 0)
            ->having('id_user', auth()->user()->id)->get();

        return count( $productos );
    }

    /**
     * Muestra todas las ofetas de un unico producto
     */

    public function show( Request $request ) {
        $data = $request->all();
        $producto = ProductoInclusionProductiva::with([
            'stocks',
            'tipoProducto' => function ( $tipo ){
                $tipo->with('imagenes');
                $tipo->with('unidad');
            },
            'ofertas' => function ( $oferta ) {
                $oferta->with(['usuario' => function ( $user ){
                    $user->with('participante');
                }]);
                $oferta->with('municipioOrigen');
                $oferta->with('municipioDestino');
                $oferta->where('estado', 'EN PROCESO');
            }
        ])->having('id', $data['id_producto'])->first();

        $producto['oferta_inicial'] = $producto['ofertas'];
        
        if($producto['oferta_inicial'] != null){
            return $producto;
        }else{
            return response()->json([
                'susses' => false,
                'message' => 'Producto no encontrado',
                'codigo' => 404
            ], 404);
        }
    }

    public function store(Request $request){
        $datosOferta = $request->all();

        $producto = ProductoInclusionProductiva::find($datosOferta['id_producto']);

        if( $producto['id_user'] === auth()->user()->id  || auth()->user()->id_role == 7){

            return response()->json([
                'susses' => false,
                'message' => 'No es posible ofertar sobre tu producto.',
                'codigo' => 403
            ], 403);

        }

        $datosOferta['direccion_destino'] = auth()->user()->direccion;
        $datosOferta['id_municipio_destino'] = auth()->user()->id_municipio;
        $datosOferta['id_user'] = auth()->user()->id;

        $stocks = StockInclusionProductiva::where('id_producto', $datosOferta['id_producto'])->get();

        $datosOferta['id_stock'] = $stocks[count($stocks)-1]['id'];

        $nuevaOferta = OfertaInclusionProductiva::create( $datosOferta );

        return response()->json([
            'susses' => true,
            'message' => 'Oferta Realizada Exitosamente',
            'oferta' => $nuevaOferta['id'],
            'vendedor' => $producto['id_user'],
            'codigo' => 200
        ], 200);
    }

    public function informacionOferta( $id ){
        $oferta = OfertaInclusionProductiva::where('id', $id)
            ->with([
                'producto' => function ( $producto ) {
                    $producto->with('tipoProducto');
                },
                'usuario',
                'municipioOrigen',
                'municipioDestino',

            ])
            ->first();
        
        if($oferta['usuario']['id'] === auth()->user()->id || auth()->user()->id_role === 2){
            $oferta['vendedor'] = User::where('id', $oferta['producto']['id_user'])->first();
            $oferta['stock_oferta'] = StockInclusionProductiva::where('id', $oferta['id_stock'])->first();
            $oferta['es_participante'] = Participante::where('id_user', $oferta['id_user'])->first() == null ? 'NO' : 'SI';
            return $oferta;
        }

        return response()->json([
            'susses' => false,
            'message' => 'Solicitud no permitida.',
            'codigo' => 403
        ], 403);


    }

    public function update( Request $request ) {
        $data = $request->all();

        // Obtiene la oferta a actualizar
        $oferta = OfertaInclusionProductiva::find($data['id_oferta']);

        if($data['nuevo_estado'] === 'ACEPTADO'){

            // Verifica si se ha cambiado la cantidad de la oferta
            if($data['cantidad_alterada'] === 'No'){
                $oferta->cantidad_venta = $data['nueva_cantidad'];
            }else{
                $oferta->cantidad_venta = $oferta->cantidad;
            }

            // Verifica si se ha cambiado el valor unitario de la oferta
            if($data['valor_alterado'] === 'No'){
                $oferta->valor_unitario_venta = $data['nuevo_valor'];
                $oferta->valor_total_venta = $data['nuevo_valor'] * $oferta->cantidad_venta;
            }else{
                $oferta->valor_unitario_venta = $oferta->valor_unitario;
                $oferta->valor_total_venta = $oferta->valor_unitario * $oferta->cantidad_venta;
            }

            $oferta->estado = 'ACEPTADO';
            $oferta->metodo_pago = $data['metodo_pago'];
            $oferta->metodo_envio = $data['metodo_envio'];
            $oferta->valor_envio = $data['valor_envio'];

            // Actualizar la oferta
            $oferta->save();

            // Actualizar el valor del stock
            $stocks = StockInclusionProductiva::where('id_producto', $data['id_producto'])->get();

            $valor_ultimo_stock = $stocks[count($stocks)-1]['valor_stock'];
            StockInclusionProductiva::create( [ 'stock' => $data['nuevo_stock'], 'valor_stock' => $valor_ultimo_stock, 'id_producto' => $data['id_producto'] ] );

            $producto = ProductoInclusionProductiva::find($data['id_producto']);
            $producto->stock_actual = $data['nuevo_stock'];
            $producto->save();
            
        }else if($data['nuevo_estado'] === 'RECHAZADO'){

            $oferta->estado = 'RECHAZADO';
            $oferta->recibido = 'NO';
            // Actualizar la oferta
            $oferta->save();

        }

        // Obtiene la información del producto actualizada
        $producto = ProductoInclusionProductiva::with([
            'stocks',
            'tipoProducto' => function ( $tipo ){
                $tipo->with('imagenes');
                $tipo->with('unidad');
            },
            'ofertas' => function ( $oferta ) {
                $oferta->with('usuario');
                $oferta->with('municipioOrigen');
                $oferta->with('municipioDestino');
                $oferta->where('estado', 'EN PROCESO');
            }
        ])->having('id', $data['id_producto'])->first();

        $producto['oferta_inicial'] = $producto['ofertas'];

        return $producto;

    }

    public function ofertasRealizadas(){
        return OfertaInclusionProductiva::with(['producto' => function ( $producto ){
                    $producto->with(['tipoProducto' => function ( $tipo ){
                        $tipo->with('imagenes');
                    }]);
                }])
                ->where('id_user', auth()->user()->id)->get();
    }

    public function respuestas(){
        return count(OfertaInclusionProductiva::where('id_user', auth()->user()->id)
                ->whereIn('estado', ['RECHAZADO', 'ACEPTADO'])
                ->where('visto', 'NO')->get());
    }

    public function visado(Request $request){
        $data = $request->all();

        $oferta = OfertaInclusionProductiva::find($data['id']);

        $oferta->visto="SI";

        $oferta->save();

        return count(OfertaInclusionProductiva::where('id_user', auth()->user()->id)
                ->whereIn('estado', ['RECHAZADO', 'ACEPTADO'])
                ->where('visto', 'NO')->get());
    }

    public function compraRecibida(Request $request){
        $data = $request->all();

        $oferta = OfertaInclusionProductiva::find($data['oferta']);

        if($oferta['estado'] !== 'RECHAZADO'){
            $oferta->latitude = $data['latitude'];
            $oferta->longitude = $data['longitude'];
            $oferta->recibido = 'SI';

            $oferta->save();

            $oferta->fecha_recibido = $oferta['updated_at'];

            $oferta->save();
        }

        return OfertaInclusionProductiva::with(['producto' => function ( $producto ){
            $producto->with(['tipoProducto' => function ( $tipo ){
                $tipo->with('imagenes');
            }]);
        }])
        ->where('id_user', auth()->user()->id)->get();
    }

}
